﻿<%@ WebHandler Language="VB" Class="DownloadHandler" %>

Imports System
Imports System.Web
Imports System.Web.SessionState
Imports System.Net
Imports System.IO
Imports System.Security.Cryptography.X509Certificates
Imports System.Net.Security

Public Class DownloadHandler : Implements IHttpHandler, IRequiresSessionState
    Private Const INVALID_FILE_URL As String = "<h2>Invalid file url !!! Please contact service provider</h2>"
    Public Sub ProcessRequest(ByVal context As HttpContext) Implements IHttpHandler.ProcessRequest
        If context.Session("fileUrl") IsNot Nothing Then
            Dim fileUrl As String = context.Session("fileUrl")
            Dim contentType As String = String.Empty
            Dim IsLocalFile As Boolean = False
            
            Try
                fileUrl = context.Server.MapPath(fileUrl)
                IsLocalFile = True
            Catch ex As Exception
                IsLocalFile = False
            End Try
            
            If IsLocalFile Then
                If (System.IO.File.Exists(fileUrl)) Then
                    'Local server file
                    Dim file As New System.IO.FileInfo(fileUrl)
                    contentType = GetContentType(System.IO.Path.GetExtension(file.FullName))
                    context.Response.ClearContent()
                    context.Response.ClearHeaders()
                    context.ClearError()
                    context.Response.Clear()
                    context.Response.Cache.SetCacheability(HttpCacheability.NoCache)
                    context.Response.Cache.SetNoStore()
                    context.Response.Cache.SetExpires(DateTime.MinValue)
                    context.Response.AddHeader("Content-Length", file.Length.ToString())
                    context.Response.AddHeader("Content-Disposition", "attachment; filename=" + file.Name)
                    context.Response.ContentType = contentType
                    context.Response.WriteFile(file.FullName)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                End If
            Else
                'External download link
                'clear response
                context.Response.ClearContent()
                context.Response.ClearHeaders()
                context.ClearError()
                context.Response.Clear()
                'validate the file url
                Dim httpUri As Uri
                If Not (Uri.TryCreate(fileUrl, UriKind.RelativeOrAbsolute, httpUri)) Then
                    context.Response.Write(INVALID_FILE_URL)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                    Exit Sub
                End If
                'Create a stream for the file
                Dim stream As Stream = Nothing

                'This controls how many bytes to read at a time and send to the client
                Dim bytesToRead As Integer = 10000

                ' Buffer to read bytes in chunk size specified above
                Dim buffer As Byte() = New [Byte](bytesToRead - 1) {}

                ' The number of bytes read
                Try
                    'Create a WebRequest to get the file
                    Dim fileReq As HttpWebRequest = DirectCast(HttpWebRequest.Create(fileUrl), HttpWebRequest)

                    'Create a response for this request
                    Dim fileResp As HttpWebResponse = DirectCast(fileReq.GetResponse(), HttpWebResponse)

                    If fileReq.ContentLength > 0 Then
                        fileResp.ContentLength = fileReq.ContentLength
                    End If

                    'Get the Stream returned from the response
                    stream = fileResp.GetResponseStream()

                    'Indicate the type of data being sent
                    context.Response.ContentType = fileResp.ContentType
                    
                    For Each key As String In fileResp.Headers.AllKeys
                        context.Response.AddHeader(key, fileResp.Headers(key))
                    Next
                
                    context.Response.AddHeader("Content-Disposition", "attachment; filename=" + HttpUtility.UrlEncode(System.IO.Path.GetFileName(fileUrl)))
                    
                    Dim length As Integer
                    Do
                        ' Verify that the client is connected.
                        If context.Response.IsClientConnected Then
                            ' Read data into the buffer.
                            length = stream.Read(buffer, 0, bytesToRead)

                            ' and write it out to the response's output stream
                            context.Response.OutputStream.Write(buffer, 0, length)

                            ' Flush the data
                            context.Response.Flush()

                            'Clear the buffer
                            buffer = New [Byte](bytesToRead - 1) {}
                        Else
                            ' cancel the download if client has disconnected
                            length = -1
                        End If
                        'Repeat until no data is read
                    Loop While length > 0
                
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                Catch wErr As WebException
                    context.Response.Write(INVALID_FILE_URL)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                Catch ufe As UriFormatException
                    context.Response.Write(INVALID_FILE_URL)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                Catch NullErr As NullReferenceException
                    context.Response.Write(NullErr.Message & " Stack: " & NullErr.StackTrace)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                Catch Err As Exception
                    context.Response.Write(Err.Message)
                    context.ApplicationInstance.CompleteRequest()
                    context.Response.End()
                Finally
                    If stream IsNot Nothing Then
                        'Close the input stream
                        stream.Close()
                    End If
                End Try

                
            End If
        End If
    End Sub
 
    Public ReadOnly Property IsReusable() As Boolean Implements IHttpHandler.IsReusable
        Get
            Return False
        End Get
    End Property

    
    Private Function GetContentType(fileExtension As String) As String
        Dim res As String
        Select Case fileExtension.Remove(0, 1).ToLower()
            Case "png"
                res = "image/png"
                Exit Select
            Case "jpeg"
                res = "image/jpg"
                Exit Select
            Case "jpg"
                res = "image/jpg"
                Exit Select
            Case "js"
                res = "application/javascript"
                Exit Select
            Case "css"
                res = "text/css"
                Exit Select
            Case "exe"
            Case "msi"
                res = "application/octet-stream"
                Exit Select
            Case "pdf"
                res = "application/pdf"
                Exit Select
            Case "xls"
            Case "xlsx"
                res = "application/vnd.ms-excel"
                Exit Select
            Case "txt"
                res = "text/plain"
                Exit Select
            Case "zip"
            Case "rar"
                res = "application/zip"
                Exit Select 
            Case Else
                res = "text/html"
                Exit Select
        End Select
        Return res

    End Function
End Class